/*
   This program reads lines from standard input (stdin)
   and filters them to standard output (stdout) but this
   program first "redirects" its standard output to be
   a child process. This creates a simple, two stage
   pipeline made up of the parent and child processes.

   If standard input is the console keyboard,
   use ^z (Control-z) to denote the end of file
   (and you must use ^z at the beginning of a line!).

   Important: Notice how this program is similar to
   the following command line pipe.

   C:\> java RemoveVowels | java Double

   In the case of the above command line, the shell
   program (cmd.exe) sets up the I/O redirection
   (the pipe) before running the two programs
   RemoveVowels.class and Double.class. In the case
   of this program, this program itself does its own
   I/O redirection (setting up the pipe) between itself
   (which acts like RemoveVowels.java) and its child
   process before running the child process.
*/
import java.util.Scanner;
import java.io.*;

public class Java7_ParentChildPipeline
{
   public static void main(String[] args) throws IOException, InterruptedException
   {
      // Create a command line for running the child.
      ProcessBuilder pb = new ProcessBuilder("java", "Double");

      // Have the child inherit the parent's standard output streams.
      pb.redirectOutput( ProcessBuilder.Redirect.INHERIT );
      pb.redirectError(  ProcessBuilder.Redirect.INHERIT );

      // Execute the child.
      Process process = pb.start();  // this throws IOException

      // Set System.out to be the standard input of the child process
      // (this is the "pipeline" between the parent and the child).
      System.setOut( new PrintStream( process.getOutputStream() ) );
    //System.setOut( new PrintStream( process.getOutputStream(), true ) );

      // Create a Scanner object to make it easier to use System.in
      Scanner scanner = new Scanner( System.in );

      // Send each line of data from the parent's stdin to the next stage.
      while ( scanner.hasNextLine() )
      {
         String oneLine = scanner.nextLine(); // read from parent's stdin
         // Filter the line of input before
         // sending it to the next stage.
         for (int i = 0; i < oneLine.length(); i++)
         {
            if ( -1 == "aeiouAEIOU".indexOf( oneLine.charAt(i) ) )
               System.out.print( oneLine.charAt(i) ); // write to stdout
         }
         System.out.println();            // write to stdout (which is the next stage)
         //System.out.flush();
      }
      System.out.close(); //THIS IS IMPORTANT (without it the child will hang)

      // Wait for the child to finish its work.
      process.waitFor();  // this throws InterruptedException
   }
}

/*
   As with the program
      Java6_ChildInheritsStdStreams_ver3.java
   from the previous folder, line 63
      System.out.flush();
   is commented out since it is not needed for this program to be
   correct, but it is needed to make this program behave nicely with
   console input and output.

   Instead of using line 63, you could use line 46, which turns
   "autoFlush" on for the PrintStream connected between the parent's
   stdout and the child's stdin. The autoFlush option has the same
   effect as calling flush() after every call to println().
*/

/*
   Compare this program with
      Java6_ParentChildPipeline_ver2.java
   and
      Java6_ParentChildPipeline_ver3.java.
   Can you remove the two lines 65
      System.out.close();
   and 68
      process.waitFor();
*/